<?php

/**
 * NOTIFICATION SYSTEM INTEGRATION EXAMPLES
 * ==========================================
 *
 * Hizi ni mifano ya jinsi ya ku-integrate notification system
 * kwenye controllers zako zilizopo.
 *
 * COPY code hizi na zi-paste kwenye controllers husika.
 */

namespace App\Http\Controllers;

use App\Http\Controllers\NotificationController;
use App\Models\Enquiry;
use App\Models\Payment;
use App\Models\LoanApplication;

/**
 * ========================================
 * EXAMPLE 1: ENQUIRY CONTROLLER
 * ========================================
 */
class EnquiryControllerExample
{
    /**
     * MFANO 1: Create new enquiry with notification
     */
    public function store(Request $request)
    {
        // Validate and create enquiry (existing code)
        $validatedData = $request->validate([...]);
        $enquiry = Enquiry::create($validatedData);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $enquiry->id,
            'enquiry_registered',
            'New Enquiry Registered',
            "Enquiry {$enquiry->check_number} registered for {$enquiry->full_name}",
            [
                'enquiry_type' => $enquiry->type,
                'member_name' => $enquiry->full_name,
                'registered_by' => auth()->user()->name,
            ]
        );

        return redirect()->back()->with('success', 'Enquiry registered successfully');
    }

    /**
     * MFANO 2: Assign enquiry to users with notification
     */
    public function assignUsersToEnquiry(Request $request, $enquiryId)
    {
        $enquiry = Enquiry::findOrFail($enquiryId);

        // Assign users (existing code)
        $enquiry->assignedUsers()->attach($request->user_ids, [
            'assigned_by' => auth()->id(),
        ]);

        // Update status
        $oldStatus = $enquiry->status;
        $enquiry->update(['status' => 'assigned']);

        // ✅ ADD THIS: Create notification for status change
        NotificationController::createNotification(
            $enquiryId,
            'status_changed',
            'Enquiry Status Updated',
            "Enquiry {$enquiry->check_number} status changed from {$oldStatus} to assigned",
            [
                'old_status' => $oldStatus,
                'new_status' => 'assigned',
                'changed_by' => auth()->user()->name,
            ]
        );

        // ✅ ADD THIS: Create notification for assignment
        NotificationController::createNotification(
            $enquiryId,
            'enquiry_assigned',
            'Enquiry Assigned',
            "Enquiry {$enquiry->check_number} has been assigned to loan officers",
            [
                'assigned_by' => auth()->user()->name,
                'assigned_to_count' => count($request->user_ids),
            ]
        );

        return redirect()->back()->with('success', 'Users assigned successfully');
    }

    /**
     * MFANO 3: Change enquiry status with notification
     */
    public function changeStatus(Request $request, $enquiryId)
    {
        $enquiry = Enquiry::findOrFail($enquiryId);
        $oldStatus = $enquiry->status;
        $newStatus = $request->status;

        // Update status (existing code)
        $enquiry->update(['status' => $newStatus]);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $enquiryId,
            'status_changed',
            'Enquiry Status Updated',
            "Enquiry {$enquiry->check_number} status changed to {$newStatus}",
            [
                'old_status' => $oldStatus,
                'new_status' => $newStatus,
                'changed_by' => auth()->user()->name,
            ]
        );

        return redirect()->back()->with('success', 'Status updated successfully');
    }
}

/**
 * ========================================
 * EXAMPLE 2: PAYMENT CONTROLLER
 * ========================================
 */
class PaymentControllerExample
{
    /**
     * MFANO 1: Initiate payment with notification
     */
    public function initiate(Request $request, $enquiryId)
    {
        $enquiry = Enquiry::findOrFail($enquiryId);

        // Create payment (existing code)
        $payment = Payment::create([
            'enquiry_id' => $enquiryId,
            'amount' => $request->amount,
            'status' => 'initiated',
            'initiated_by' => auth()->id(),
        ]);

        // Update enquiry status
        $enquiry->update(['status' => 'payment_initiated']);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $enquiryId,
            'payment_initiated',
            'Payment Initiated',
            "Payment of Tsh " . number_format($request->amount) . " has been initiated for {$enquiry->full_name}",
            [
                'amount' => $request->amount,
                'payment_id' => $payment->id,
                'initiated_by' => auth()->user()->name,
                'check_number' => $enquiry->check_number,
            ]
        );

        return redirect()->back()->with('success', 'Payment initiated successfully');
    }

    /**
     * MFANO 2: Approve payment (General Manager action)
     */
    public function approve($paymentId)
    {
        $payment = Payment::findOrFail($paymentId);
        $enquiry = $payment->enquiry;

        // Update payment status (existing code)
        $payment->update([
            'status' => 'approved',
            'approved_by' => auth()->id(),
            'approved_at' => now(),
        ]);

        $enquiry->update(['status' => 'approved']);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $payment->enquiry_id,
            'payment_approved',
            'Payment Approved',
            "Payment of Tsh " . number_format($payment->amount) . " has been approved",
            [
                'amount' => $payment->amount,
                'approved_by' => auth()->user()->name,
                'check_number' => $enquiry->check_number,
            ]
        );

        return redirect()->back()->with('success', 'Payment approved successfully');
    }

    /**
     * MFANO 3: Reject payment
     */
    public function reject(Request $request, $paymentId)
    {
        $payment = Payment::findOrFail($paymentId);
        $enquiry = $payment->enquiry;

        // Update payment status (existing code)
        $payment->update([
            'status' => 'rejected',
            'rejected_by' => auth()->id(),
            'rejection_reason' => $request->reason,
        ]);

        $enquiry->update(['status' => 'rejected']);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $payment->enquiry_id,
            'payment_rejected',
            'Payment Rejected',
            "Payment of Tsh " . number_format($payment->amount) . " has been rejected",
            [
                'amount' => $payment->amount,
                'rejected_by' => auth()->user()->name,
                'reason' => $request->reason,
                'check_number' => $enquiry->check_number,
            ]
        );

        return redirect()->back()->with('success', 'Payment rejected');
    }

    /**
     * MFANO 4: Mark payment as paid (Accountant action)
     */
    public function markAsPaid($paymentId)
    {
        $payment = Payment::findOrFail($paymentId);
        $enquiry = $payment->enquiry;

        // Update payment status (existing code)
        $payment->update([
            'status' => 'paid',
            'paid_by' => auth()->id(),
            'paid_at' => now(),
        ]);

        $enquiry->update(['status' => 'paid']);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $payment->enquiry_id,
            'payment_paid',
            'Payment Completed',
            "Payment of Tsh " . number_format($payment->amount) . " has been completed",
            [
                'amount' => $payment->amount,
                'paid_by' => auth()->user()->name,
                'check_number' => $enquiry->check_number,
            ]
        );

        return redirect()->back()->with('success', 'Payment marked as paid');
    }
}

/**
 * ========================================
 * EXAMPLE 3: LOAN CONTROLLER
 * ========================================
 */
class LoanControllerExample
{
    /**
     * MFANO 1: Process loan application (Loan Officer action)
     */
    public function process($loanApplicationId)
    {
        $loanApplication = LoanApplication::findOrFail($loanApplicationId);
        $enquiry = $loanApplication->enquiry;

        // Update loan status (existing code)
        $loanApplication->update([
            'status' => 'processed',
            'processed_by' => auth()->id(),
            'processed_at' => now(),
        ]);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $loanApplication->enquiry_id,
            'loan_processed',
            'Loan Application Processed',
            "Loan application for Tsh " . number_format($loanApplication->loan_amount) . " has been processed",
            [
                'loan_amount' => $loanApplication->loan_amount,
                'loan_type' => $loanApplication->loan_type,
                'processed_by' => auth()->user()->name,
                'check_number' => $enquiry->check_number,
            ]
        );

        return redirect()->back()->with('success', 'Loan processed successfully');
    }

    /**
     * MFANO 2: Approve loan (General Manager action)
     */
    public function approve($loanApplicationId)
    {
        $loanApplication = LoanApplication::findOrFail($loanApplicationId);
        $enquiry = $loanApplication->enquiry;

        // Update loan status (existing code)
        $loanApplication->update([
            'status' => 'approved',
            'approved_by' => auth()->id(),
            'approved_at' => now(),
        ]);

        $enquiry->update(['status' => 'approved']);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $loanApplication->enquiry_id,
            'loan_approved',
            'Loan Application Approved',
            "Loan application for Tsh " . number_format($loanApplication->loan_amount) . " has been approved",
            [
                'loan_amount' => $loanApplication->loan_amount,
                'loan_type' => $loanApplication->loan_type,
                'approved_by' => auth()->user()->name,
                'check_number' => $enquiry->check_number,
            ]
        );

        return redirect()->back()->with('success', 'Loan approved successfully');
    }

    /**
     * MFANO 3: Reject loan
     */
    public function reject(Request $request, $loanApplicationId)
    {
        $loanApplication = LoanApplication::findOrFail($loanApplicationId);
        $enquiry = $loanApplication->enquiry;

        // Update loan status (existing code)
        $loanApplication->update([
            'status' => 'rejected',
            'rejected_by' => auth()->id(),
            'rejection_reason' => $request->reason,
        ]);

        $enquiry->update(['status' => 'rejected']);

        // ✅ ADD THIS: Create notification
        NotificationController::createNotification(
            $loanApplication->enquiry_id,
            'loan_rejected',
            'Loan Application Rejected',
            "Loan application for Tsh " . number_format($loanApplication->loan_amount) . " has been rejected",
            [
                'loan_amount' => $loanApplication->loan_amount,
                'loan_type' => $loanApplication->loan_type,
                'rejected_by' => auth()->user()->name,
                'reason' => $request->reason,
                'check_number' => $enquiry->check_number,
            ]
        );

        return redirect()->back()->with('success', 'Loan rejected');
    }
}

/**
 * ========================================
 * QUICK REFERENCE
 * ========================================
 *
 * Notification Types Available:
 * - enquiry_registered
 * - enquiry_assigned
 * - status_changed
 * - payment_initiated
 * - payment_approved
 * - payment_rejected
 * - payment_paid
 * - loan_processed
 * - loan_approved
 * - loan_rejected
 *
 * Basic Syntax:
 * NotificationController::createNotification(
 *     $enquiryId,           // Enquiry ID (required)
 *     'notification_type',  // Type from list above
 *     'Title',              // Short title
 *     'Full message',       // Detailed message
 *     [                     // Meta data (optional)
 *         'key' => 'value',
 *     ]
 * );
 */
